<?php

namespace App\Http\Controllers;

use App\Models\Customers;
use App\Models\CustomersPlans;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Str;

class AuthController extends Controller
{
    public function signup(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'first_name'        => 'required|string|max:255',
                'mobile'            => 'required|numeric|unique:customers,mobile',
                'email'             => 'required|email|unique:customers,email',
                'password'          => 'required|min:6',
                'confirm_password'  => 'required|same:password',
                'terms'             => 'accepted',
            ], [
                'first_name.required' => 'First name is required.',
                'first_name.string'   => 'First name must be a valid string.',
                'first_name.max'      => 'First name cannot exceed 255 characters.',

                'mobile.required'     => 'Mobile number is required.',
                'mobile.numeric'      => 'Mobile number must be numeric.',
                'mobile.unique'       => 'Mobile number is already taken.',

                'email.required'      => 'Email is required.',
                'email.email'         => 'Email must be a valid email address.',
                'email.unique'        => 'Email is already taken.',

                'password.required'   => 'Password is required.',
                'password.min'        => 'Password must be at least 6 characters.',

                'confirm_password.required' => 'Confirm Password is required.',
                'confirm_password.same'     => 'Passwords do not match.',

                'terms.accepted'      => 'You must accept the terms.',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => $validator->errors()->first(),
                    'errors'  => $validator->errors() // optional: return all errors
                ], 422);
            }

            $check_customer = Customers::where('email', $request->email)->count();
            if ($check_customer > 0) {
                return response()->json(['status' => 'error', 'message' => 'You are already registered, please verify yourself!'], 400);
            }

            $customer_order = Customers::max('position_order');
            $position_order = ($customer_order !== null) ? $customer_order + 1 : 1;

            $customer = new Customers();
            $customer->position_order = $position_order;
            $customer->first_name = $request->first_name;
            $customer->last_name = $request->last_name;
            $customer->mobile = $request->mobile;
            $customer->email = $request->email;
            $customer->password = Hash::make($request->password);
            $customer->terms = $request->terms;
            $customer->save();

            // $token = $customer->createToken('customer-token')->plainTextToken;
            $token = Crypt::encrypt($customer->id);

            return response()->json([
                'status' => 'success',
                'message' => 'Signup Successful!',
                'access_token' => $token,
                'customer' => $customer,
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        }
    }

    public function login(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required|email',
                'password' => 'required|string',
            ], [
                'email.required'      => 'Email is required.',
                'email.email'         => 'Email must be a valid email address.',

                'password.required'   => 'Password is required.',
                'password.string'     => 'Password must be in string format.',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => $validator->errors()->first(),
                    'errors'  => $validator->errors()
                ], 422);
            }

            $customer = Customers::where('email', $request->email)->first();

            if (!$customer) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'You are not registered, please make an account!',
                ], 401); // Unauthorized
            }

            if ($customer->status == 'deactive') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Please verify your account to login, contact to Administrator!',
                ], 401); // Unauthorized
            }

            if (!Hash::check($request->password, $customer->password)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid credentials',
                ], 401); // Unauthorized
            }

            $customer_active_plan = null;
            if ($customer->active_plan_id != 0) {
                $customer_active_plan = CustomersPlans::with([
                    'plan:id,plan_name,plan_price,plan_price_base,plan_storage_limit'
                ])
                    ->select('id', 'plan_status', 'start_date', 'expiry_date', 'plan_id')
                    ->where([
                        'customer_id' => $customer->id,
                        'plan_id' => $customer->active_plan_id,
                    ])
                    ->first();
            }

            $token = Crypt::encrypt($customer->id);

            return response()->json([
                'status' => 'success',
                'access_token' => $token,
                'token_type' => 'Bearer',
                'customer' => $customer,
                'active_plan' => $customer_active_plan,
            ], 200); // OK

        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422); // Unprocessable Entity
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Something went wrong',
            ], 500); // Internal Server Error
        }
    }

    public function getCustomerProfile(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'customer_enc' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            $customer_id = Crypt::encrypt($request->customer_enc);
            $customer = Customers::find($customer_id);

            if (!$customer) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Data not found!',
                ], 404); // Not found
            }

            $customer_active_plan = null;
            if ($customer->active_plan_id != 0) {
                $customer_active_plan = CustomersPlans::with([
                    'plan:id,plan_name,plan_price,plan_price_base,plan_storage_limit'
                ])
                    ->select('id', 'plan_status', 'start_date', 'expiry_date', 'plan_id')
                    ->where([
                        'customer_id' => $customer->id,
                        'plan_id' => $customer->active_plan_id,
                    ])
                    ->first();
            }

            return response()->json([
                'status' => 'success',
                'customer' => $customer,
                'active_plan' => $customer_active_plan,
            ], 200); // OK

        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422); // Unprocessable Entity
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Something went wrong',
            ], 500); // Internal Server Error
        }
    }

    public function checkEmailAndSendLink(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required|email'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            $customer = Customers::where('email', $request->email)->where('status', 'active')->first();

            if (!$customer) {
                return response()->json([
                    'message' => 'Account not found, please register or verify yourself.',
                ], 404);
            }

            $status = Password::broker('customer')->sendResetLink(
                $request->only('email')
            );

            if ($status === Password::RESET_LINK_SENT) {
                return response()->json([
                    'message' => 'A reset link has been sent to your E-Mail address. Please check your inbox.',
                ], 200);
            }

            return response()->json([
                'message' => 'Failed to send reset link. Please try again later.',
            ], 500);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An unexpected error occurred.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function checkResetLink(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required|email',
                'token' => 'required|string'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            $customer = Customers::where('email', $request->email)->where('status', 'active')->first();

            if (!$customer) {
                return response()->json([
                    'message' => 'Invalid reset link. Please request a new one.',
                ], 404);
            }

            $record = DB::table('password_reset_tokens')
                ->where('email', $request->email)
                ->first();

            if (!$record || !Hash::check($request->token, $record->token)) {
                return response()->json([
                    'message' => 'Invalid or expired reset link.',
                ], 404);
            }

            // Optional: check token expiration (10 minutes)
            if (Carbon::parse($record->created_at)->addMinutes(10)->isPast()) {
                return response()->json([
                    'message' => 'Reset link has expired. Please request a new one.',
                ], 404);
            }

            return response()->json([
                'message' => 'Reset link is valid.',
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An unexpected error occurred while verifying the link.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function resetPassword(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required|email',
                'token' => 'required|string',
                'password' => 'required|string|min:6|confirmed', // expects password_confirmation
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            $current_customer = Customers::where('email', $request->email)->where('status', 'active')->first();

            if (!$current_customer) {
                return response()->json([
                    'message' => 'Account is not registered or verified. Please register or verify yourself!',
                ], 404);
            }

            $status = Password::broker('customer')->reset(
                $request->only('email', 'password', 'password_confirmation', 'token'),
                function ($customer, $password) {
                    $customer->password = Hash::make($password);
                    $customer->save();

                    $customer->setRememberToken(Str::random(60));
                    $customer->save();
                }
            );

            if ($status === Password::PASSWORD_RESET) {
                return response()->json([
                    'message' => 'Password has been reset successfully.',
                ], 200);
            }

            return response()->json([
                'message' => $status === Password::INVALID_TOKEN
                    ? 'This reset link has already been used or expired. Please request a new one.'
                    : __($status),
            ], 400);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An unexpected error occurred.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }
}
