<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Mail\SendOtpMail;
use App\Models\Admin\Company;
use App\Models\Admin\User;
use App\Models\CustomersPlansTransactions;
use App\Models\Files;
use App\Models\Modules;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Session;

class AuthController extends Controller
{
    private function sendOtp($email)
    {
        try {
            $user = User::where('email', $email)->first();

            if (!$user) {
                return false; // user not found
            }

            $otp = rand(100000, 999999);

            $user->update([
                'login_otp' => $otp,
                'login_otp_expires_at' => Carbon::now()->addMinutes(10),
            ]);

            Mail::to($user->email)->send(new SendOtpMail($otp));
            return true;
        } catch (\Exception $e) {
            Log::error('OTP Sending Failed: ' . $e->getMessage());
            return false;
        }
    }


    public function register(Request $request)
    {
        if (Auth::check()) {
            return redirect()->route('dashboard');
        }

        if ($request->isMethod('post')) {
            $request->validate([
                'first_name' => 'required|string',
                'mail' => 'required|email|max:255',
                'pwd' => 'required|string',
                'confirm_pwd' => 'required|string|same:pwd',
            ], [
                'first_name.required' => 'First Name is required!',
                'first_name.string' => 'First Name should be string format!',
                'mail.required' => 'E-Mail is required!',
                'mail.email' => 'Invalid E-Mail!',
                'pwd.required' => 'Password is required!',
                'confirm_pwd.required' => 'Confirm Password is required!',
                'confirm_pwd.same' => 'Password confirmation should be matched!',
            ]);

            // dd(Hash::make($request->pwd));
            // dd(Crypt::encrypt($request->pwd));

            User::create([
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'email' => $request->mail,
                'password' => Hash::make($request->pwd),
            ]);

            return redirect()->route('login')->with('success', 'Registration successful. Please log in.');
        }

        return view('admin.register');
    }

    public function login(Request $request)
    {
        if (Auth::check()) {
            return redirect()->route('dashboard');
        }

        if ($request->isMethod('post')) {
            $request->validate([
                'mail' => 'required|email|max:255',
                'pwd' => 'required|string',
            ], [
                'mail.required' => 'E-Mail is required!',
                'mail.email' => 'Invalid E-Mail!',
                'pwd.required' => 'Password is required!',
            ]);

            // dd(Hash::make($request->pwd));

            $user = User::where('email', $request->mail)->first();
            if (!$user) {
                return back()->withErrors(['error' => 'No account found with this E-Mail!'])->withInput();
            }

            if (!Hash::check($request->pwd, $user->password)) {
                return back()->withErrors(['error' => 'Invalid E-Mail or Password!'])->withInput();
            }

            // $request->session()->regenerate();
            Session::put('admin_email', $request->mail);
            $sendOtpResponse = $this->sendOtp($request->mail);
            if (!$sendOtpResponse) {
                return back()->withErrors(['error' => 'Error in sending the OTP!'])->withInput();
            }

            return redirect()->route('verify.otp');
        }

        return view('admin.login');
    }

    public function verifyOtp(Request $request)
    {
        if (Auth::check()) {
            return redirect()->route('dashboard');
        }

        if ($request->isMethod('post')) {
            $request->validate([
                'login_otp' => 'required',
            ], [
                'login_otp.required' => 'OTP is required!',
            ]);

            $email = Session::get('admin_email');
            Session::forget('admin_email');
            $user = User::where('email', $email)->first();
            if (!$user) {
                return back()->withErrors(['error' => 'No account found with this E-Mail!'])->withInput();
            }

            if ($request->login_otp != $user->login_otp) {
                return back()->withErrors(['error' => 'Invalid OTP!'])->withInput();
            }

            if (Carbon::now()->greaterThan($user->login_otp_expires_at)) {
                return back()->withErrors(['error' => 'OTP has expired. Please request a new one.'])->withInput();
            }

            $request->session()->regenerate();
            Auth::login($user);

            $user->update([
                'login_otp' => null,
                'login_otp_expires_at' => null,
            ]);

            // Set extra session data
            Session::put('name', $user->first_name . ' ' . $user->last_name);
            Session::put('role', $user->role);
            Session::put('profile_image', $user->image);

            $company = Company::findOrFail(1, ['company_logo', 'company_name']);
            if ($company) {
                Session::put('company_name', $company->company_name);
                Session::put('company_logo', $company->company_logo);
            } else {
                Log::warning('Company record not found.');
            }

            return redirect()->route('dashboard');
        }

        return view('admin.verify-otp');
    }

    public function resendOtp()
    {
        if (Auth::check()) {
            $user = Auth::user();
            $email = $user->email;
        } else {
            $email = Session::get('admin_email');
        }

        if (!$email) {
            return response()->json([
                'status' => 'error',
                'message' => 'Data is expired. Please login again.'
            ], 400);
        }

        $sendOtpResponse = $this->sendOtp($email);

        if (!$sendOtpResponse) {
            return response()->json([
                'status' => 'error',
                'message' => 'Error in sending the OTP!'
            ], 500);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'OTP has been resent successfully!'
        ], 200);
    }

    public function dashboard()
    {
        // // Uploaded files
        // $uploaded_count = Files::where(['file_status' => 'created', 'status' => 'added'])
        //     ->whereHas('module', function ($q) {
        //         $q->where('status', 'uploaded');
        //     })
        //     ->count();

        // // Downloaded files
        // $downloaded_count = Files::where(['file_status' => 'downloaded', 'status' => 'deleted'])
        //     ->whereHas('module', function ($q) {
        //         $q->where('status', 'downloaded_expired');
        //     })
        //     ->count();

        // // Partially downloaded files
        // $partially_downloaded_count = Files::where(['file_status' => 'downloaded', 'status' => 'deleted'])
        //     ->whereHas('module', function ($q) {
        //         $q->where('status', 'partially_downloaded');
        //     })
        //     ->count();

        // // Active links (not expired)
        // $activated_link_count = Modules::whereNotIn('status', ['expired', 'downloaded_expired'])->count();

        // // Expired links
        // $expired_link_count = Modules::whereIn('status', ['expired', 'downloaded_expired'])->count();

        // // Total files (sum of all types)
        // $files_count = $uploaded_count + $downloaded_count + $partially_downloaded_count + $activated_link_count + $expired_link_count;

        $data = [
            'files_count'                => 0,
            'uploaded_count'             => 0,
            'downloaded_count'           => 0,
            'partially_downloaded_count' => 0,
            'activated_link_count'       => 0,
            'expired_link_count'         => 0,
        ];

        $currentPage = "dashboard";
        return view('admin.dashboard', compact('data', 'currentPage'));
    }

    public function fileTransferData()
    {
        $uploaded_count = Files::where(['file_status' => 'created', 'status' => 'added'])
            ->whereHas('module', fn($q) => $q->where('status', 'uploaded'))
            ->count();

        $downloaded_count = Files::where(['file_status' => 'downloaded', 'status' => 'deleted'])
            ->whereHas('module', fn($q) => $q->where('status', 'downloaded_expired'))
            ->count();

        $partially_downloaded_count = Files::where(['file_status' => 'downloaded', 'status' => 'deleted'])
            ->whereHas('module', fn($q) => $q->where('status', 'partially_downloaded'))
            ->count();

        $activated_link_count = Modules::whereNotIn('status', ['expired', 'downloaded_expired'])->count();
        $expired_link_count   = Modules::whereIn('status', ['expired', 'downloaded_expired'])->count();

        $labels = ['Uploaded', 'Downloaded', 'Partially Downloaded', 'Active Links', 'Expired'];
        $series = [
            $uploaded_count,
            $downloaded_count,
            $partially_downloaded_count,
            $activated_link_count,
            $expired_link_count,
        ];

        $colors = [
            '#696cff', // primary
            '#71dd37', // success
            '#ffab00', // warning
            '#03c3ec', // info
            '#ff4b2d', // danger
        ];

        return response()->json([
            'labels' => $labels,
            'series' => $series,
            'colors' => $colors,
        ]);
    }

    public function uploadedChartData()
    {
        $uploadedData = Files::selectRaw('MONTH(created_at) as month, COUNT(*) as total')
            ->where('status', 'added')
            ->whereHas('module', fn($q) => $q->where('status', 'uploaded'))
            ->whereYear('created_at', now()->year)
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        $months = collect(range(1, 12))->map(fn($m) => Carbon::create()->month($m)->format('M'))->toArray();
        $series = collect(range(1, 12))->map(fn($m) => $uploadedData->firstWhere('month', $m)->total ?? 0)->toArray();

        $uploadedCount = $uploadedData->sum('total');

        return response()->json([
            'uploaded_count' => $uploadedCount,
            'months' => $months,
            'series' => $series
        ]);
    }

    public function weeklyUploadedData()
    {
        $now = now();
        $startOfWeek = $now->copy()->startOfWeek();
        $endOfWeek = $now->copy()->endOfWeek();

        $startOfLastWeek = $now->copy()->subWeek()->startOfWeek();
        $endOfLastWeek = $now->copy()->subWeek()->endOfWeek();

        $thisWeekCount = Files::where('status', 'added')
            ->whereHas('module', fn($q) => $q->where('status', 'uploaded'))
            ->whereBetween('created_at', [$startOfWeek, $endOfWeek])
            ->count();

        $lastWeekCount = Files::where('status', 'added')
            ->whereHas('module', fn($q) => $q->where('status', 'uploaded'))
            ->whereBetween('created_at', [$startOfLastWeek, $endOfLastWeek])
            ->count();

        $difference = $thisWeekCount - $lastWeekCount;
        $percentage = $lastWeekCount > 0 ? round(($difference / $lastWeekCount) * 100, 2) : 0;

        return response()->json([
            'this_week' => $thisWeekCount,
            'last_week' => $lastWeekCount,
            'difference' => $difference,
            'percentage' => $percentage,
        ]);
    }

    public function downloadedChartData()
    {
        $downloadedData = Files::selectRaw('MONTH(created_at) as month, COUNT(*) as total')
            ->where(['file_status' => 'downloaded', 'status' => 'deleted'])
            ->whereHas('module', fn($q) => $q->where('status', 'downloaded_expired'))
            ->whereYear('created_at', now()->year)
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        $months = collect(range(1, 12))->map(fn($m) => Carbon::create()->month($m)->format('M'))->toArray();
        $series = collect(range(1, 12))->map(fn($m) => $downloadedData->firstWhere('month', $m)->total ?? 0)->toArray();

        return response()->json([
            'months' => $months,
            'series' => $series,
            'total' => array_sum($series),
        ]);
    }

    public function partiallyDownloadedChartData()
    {
        $downloadedData = Files::selectRaw('MONTH(created_at) as month, COUNT(*) as total')
            ->where(['file_status' => 'downloaded', 'status' => 'deleted'])
            ->whereHas('module', fn($q) => $q->where('status', 'partially_downloaded'))
            ->whereYear('created_at', now()->year)
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        $months = collect(range(1, 12))->map(fn($m) => Carbon::create()->month($m)->format('M'))->toArray();
        $series = collect(range(1, 12))->map(fn($m) => $downloadedData->firstWhere('month', $m)->total ?? 0)->toArray();

        return response()->json([
            'months' => $months,
            'series' => $series,
            'total' => array_sum($series),
        ]);
    }

    public function expiredChartData()
    {
        $downloadedData = Files::selectRaw('MONTH(created_at) as month, COUNT(*) as total')
            ->whereIn('file_status', ['downloaded', 'removed'])
            ->where('status', 'deleted')
            ->whereHas('module', fn($q) => $q->whereIn('status', ['downloaded_expired', 'expired']))
            ->whereYear('created_at', now()->year)
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        $months = collect(range(1, 12))->map(fn($m) => Carbon::create()->month($m)->format('M'))->toArray();
        $series = collect(range(1, 12))->map(fn($m) => $downloadedData->firstWhere('month', $m)->total ?? 0)->toArray();

        return response()->json([
            'months' => $months,
            'series' => $series,
            'total' => array_sum($series),
        ]);
    }

    public function getTransactionsData(Request $request)
    {
        $filter = $request->query('filter', 'thirty_days');

        $query = CustomersPlansTransactions::with('plan:id,plan_name');

        // Filter logic
        if ($filter === 'three_months') {
            $query->where('created_at', '>=', now()->subMonths(3));
        } elseif ($filter === 'year') {
            $query->whereYear('created_at', now()->year);
        } else { // Default = 30 days
            $query->where('created_at', '>=', now()->subDays(30));
        }

        $transactions = $query->latest()->take(10)->get()->map(function ($t) {
            return [
                'type' => 'purchase',
                'platform' => ($t->payment_gateway == 'RAZOR_PAY' ? 'Razorpay' : ''),
                'title' => $t->plan->plan_name,
                'amount' => number_format($t->amount, 2),
                'currency' => $t->currency ?? 'INR',
                'status' => $t->transaction_status,
            ];
        });

        return response()->json(['transactions' => $transactions]);
    }

    public function editProfile(Request $request)
    {
        if ($request->isMethod('post')) {
            $request->validate([
                'firstName' => 'required|string',
                'email' => 'required|string',
                'phone' => 'required|integer',
            ], [
                'firstName.required' => 'First Name is required!',
                'firstName.string' => 'First Name should be string format!',
                'email.required' => 'E-Mail is required!',
                'email.email' => 'Invalid E-Mail!',
                'phone.required' => 'Phone is required!',
                'phone.integer' => 'Phone should be integer type!',
            ]);

            $id = Auth::id();
            $profile = User::findOrFail($id);
            $profile->first_name = $request->firstName;
            $profile->last_name = $request->lastName;
            $profile->email = $request->email;
            $profile->phone = $request->phone;

            if ($request->hasFile('profile_image')) {
                $path = 'images/users/';
                $filePath = $this->storeImage($request->file("profile_image"), $path, $profile->image);
                $profile->image = $filePath;
            }

            if ($profile->save()) {
                $request->session()->flash('success', 'Profile is updated Successfully!');
            } else {
                $request->session()->flash('error', 'Updation Error!');
            }

            return redirect()->route('edit.profile');
        }

        $id = Auth::id();
        $profile = User::findOrFail($id);
        return view('admin.profile-ops', compact('profile'));
    }

    public function changePassword(Request $request)
    {
        $id = Auth::id();
        $user = User::findOrFail($id);
        if (! $user) {
            return redirect()->route('login')->with('error', 'Please login first.');
        }

        if ($request->isMethod('post')) {
            // Validation
            $request->validate([
                'old_pwd' => 'required',
                'new_pwd' => 'required|min:6|confirmed',
            ]);

            // Old password check
            if (!Hash::check($request->old_pwd, $user->password)) {
                return back()->with('error', 'Old password incorrect.');
            }

            // $sendOtpResponse = $this->sendOtp($user->email);
            // if (!$sendOtpResponse) {
            //     return back()->withErrors(['error' => 'Error in sending the OTP!'])->withInput();
            // }

            // Session::put('old_admin_password', $request->old_pwd);
            // Session::put('new_admin_password', $request->new_pwd);
            // return redirect()->route('verify.change.password.otp');

            $user->password = Hash::make($request->new_pwd);
            $user->login_otp = null;
            $user->login_otp_expires_at = null;
            $user->save();

            // $old_password = Session::get('old_admin_password');
            // Session::forget(['new_admin_password', 'old_admin_password']);

            // Auth::logoutOtherDevices($request->new_pwd);

            return redirect()->route('logout')->with('success', 'Password updated successfully!');
        }

        return view('admin.change-password');
    }

    public function changePasswordVerifyOtp(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        if ($request->isMethod('post')) {
            $request->validate([
                'login_otp' => 'required',
            ], [
                'login_otp.required' => 'OTP is required!',
            ]);

            $id = Auth::id();
            $user = User::findOrFail($id);
            if (!$user) {
                return back()->withErrors(['error' => 'No account found with this E-Mail!'])->withInput();
            }

            if ($request->login_otp != $user->login_otp) {
                return back()->withErrors(['error' => 'Invalid OTP!'])->withInput();
            }

            if (Carbon::now()->greaterThan($user->login_otp_expires_at)) {
                return back()->withErrors(['error' => 'OTP has expired. Please request a new one.'])->withInput();
            }

            $new_password = trim(Session::get('new_admin_password'));
            if (!$new_password) {
                return back()->with('error', 'Session expired. Try again.');
            }

            $user->update([
                'password' => Hash::make($new_password),
                'login_otp' => null,
                'login_otp_expires_at' => null
            ]);

            $old_password = Session::get('old_admin_password');
            Session::forget(['new_admin_password', 'old_admin_password']);

            Auth::logoutOtherDevices($old_password);

            return redirect()->route('logout')->with('success', 'Password updated successfully!');
        }

        return view('admin.verify-change-password-otp');
    }

    public function logout(Request $request)
    {
        Auth::logout();

        // Invalidate the session
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login')->with('success', 'You have been logged out successfully.');
    }
}
